/* This file is part of KNemo
   Copyright (C) 2010 John Stamp <jstamp@users.sourceforge.net>

   KNemo is free software; you can redistribute it and/or modify
   it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   KNemo is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "global.h"
#include "statisticsmodel.h"
#include "xmlstorage.h"
#include "commonstorage.h"

#include <QDomNode>
#include <QFile>
#include <KCalendarSystem>

// xml storage
static const QLatin1String doc_name("statistics");
static const QLatin1String attrib_calendar("calendar");
static const QLatin1String attrib_updated("lastUpdated");
static const QLatin1String attrib_rx("rxBytes");
static const QLatin1String attrib_tx("txBytes");


XmlStorage::XmlStorage()
{
}

void XmlStorage::loadGroup( StorageData *sd, const QDomElement& parentItem,
    StatisticsModel* statistics )
{
    QDomNode n = parentItem.namedItem( periods.at( statistics->periodType() ) + QLatin1Char('s') );
    if ( !n.isNull() )
    {
        QDomNode node = n.firstChild();
        while ( !node.isNull() )
        {
            QDomElement element = node.toElement();
            if ( !element.isNull() )
            {
                QDate date;
                QTime time;

                int year = element.attribute( periods.at( KNemoStats::Year ) ).toInt();
                int month = element.attribute( periods.at( KNemoStats::Month ), QLatin1String("1") ).toInt();
                int day = element.attribute( periods.at( KNemoStats::Day ), QLatin1String("1") ).toInt();
                sd->calendar->setDate( date, year, month, day );

                if ( date.isValid() )
                {
                    switch ( statistics->periodType() )
                    {
                        case KNemoStats::Hour:
                            time = QTime( element.attribute( periods.at( KNemoStats::Hour ) ).toInt(), 0 );
                            break;
                        default:
                            ;;
                    }

                    int entryIndex = statistics->createEntry( QDateTime( date, time ) );
                    statistics->setTraffic( entryIndex, element.attribute( attrib_rx ).toULongLong(), element.attribute( attrib_tx ).toULongLong() );
                }
            }
            node = node.nextSibling();
        }
        statistics->sort( 0 );
        // Make sure the ids are in order after a sort
        for ( int i = 0; i < statistics->rowCount(); ++i )
        {
            statistics->setId( i, i );
        }
    }
}

bool XmlStorage::loadStats( QString name, StorageData *sd, QHash<int, StatisticsModel*> *models )
{
    QDomDocument doc( doc_name );
    QFile file( generalSettings->statisticsDir.absoluteFilePath( statistics_prefix + name ) );

    if ( !file.open( QIODevice::ReadOnly ) )
        return false;
    if ( !doc.setContent( &file ) )
    {
        file.close();
        return false;
    }
    file.close();

    QDomElement root = doc.documentElement();

    // If unknown, empty, or still using calendar names, use the default calendar
    sd->calendar = KCalendarSystem::create( KLocale::QDateCalendar );
    foreach( StatisticsModel * s, *models )
    {
        s->setCalendar( sd->calendar );
        loadGroup( sd, root, s );
    }

    sd->lastSaved = root.attribute( attrib_updated ).toUInt();

    return true;
}


